// BSNP - Bayesian Genotype caller from SAMTOOLS Pileup files.
//	Requires an input file generated by samtools Pileup -cv, which includes qualities and explicit reference
//		genome nucleotides at eash position. For a list of Please see PrintUsage() function and 
//		InputArgs object. This evolved quickly from project code. My apologies for the prolific use of structs
//		and other encapsulation unfriendly codeing bits.
//
// Simply Compile with g++ -O3 -o BSNP *.cpp
//
//	Written by Brad Gulko, Siepel Lab, Cornell University department of Biological Statisticational Biology
//		http://compgen.bscb.cornell.edu/~acs/
//
//	Copyright 2010 Brad Gulko & Cornell University.
//		bgulko@cs.cornell.edu / bg279@cornell.edu.
//
//	V2.05a - 2010-05 First complete release
//
//	V2.10  - 2010-07 Added:
//		assymetric sequencing technologyy dependentbase-miscall handeling
//		updated default proprs generated from 8-Genome Population Genetics resequencing.
//		numerous comments and code cleanup.
//
//	V2.11  - 2010-07-12 Added:
//		enhanced summary data dump
//		reset defaults to non-informative prior (except for p0 = .001)
//
//	V2.13  - 2010-07-20
//		fixed pathological interrraction between correlated error adjustment and 
//			technology specific non-uniform read error adjustments.
//
//	V2.14  - 2010-07-21
//		fixed bug in gathering of summary statistics. DataNLL values were 10* too large
//			(NLL added once per geno, rather than just once).
//
//	V2.15  - 2011-10-12
//		Cleanup code, replicate a fix to the indel parser (allow indel >99 chars)
//              Commenting, improving Usage Documentation etc. Fixed a typo in the the
//		technology specific error model for 454 sequencing.
//
//	V2.16  - 2011-10-14
//		Added support for fully custom priors. use the -ip option, create a file with line containing AA .15\n AC .2 etc.
//		#starts a comment and redundant whitespace is ignored. All 10 must be specified. Haploid priors are 
//		inferred from diploid distribution and both are normalized.
//
//	V2.17  - 2012-05-16 A few cleanups for release, 
//		augment print-usage to include priors file.
//		Clean up compiler -pedantic warnings

// Microsoft requries this to supress warnings for stdio functions.
#define _CRT_SECURE_NO_WARNINGS

#include <string>
using std::string;

#include <vector>
using std::vector;

#define VERSION "2.17"

#include "LogProb.h"				// class for handeling probabilities in logspace
#include "PileupReader.h"			// class to extract info from samtools pileup command
#include "ShortReadGenotype.h"

typedef unsigned long ulong;

#include "SummaryData.h"
#include "NonSNPBuffer.h"
#include "InputArgs.h"

// debugging variables
double _debug_SNP_found;
double _debug_nSNP_found;
double _debug_SNP_written;
double _debug_LinesRead;
double _debug_LinesDiscarded;

// Handy macro for closing files.
#define SAFE_CLOSE( a )	{ if ( (a!=NULL) && (a!=stderr) && (a!=stdout) ) { fclose(a); }; a = NULL; }

bool	PrintSNP(FILE *fout, const PileupReader &Reader, const ShortReadGenotype &Caller, const InputArgs &Args );
bool	PrintSharedInfo(FILE *fout, const InputArgs &I, const PileupReader &Reader, ShortReadGenotype &Caller );
void	GatherNonSNP(FILE *fout, NonSNPBuffer &Buf, const PileupReader &Reader, const ShortReadGenotype &Caller );
void	FlushNonSNP(FILE *fout, NonSNPBuffer &Buf);
void	PrintUsage( const InputArgs *A = NULL, FILE *f = NULL );
void	LocalAbort(long Code, const char *ErrMsg, const char *ErrContext = NULL, long *ErrValL = NULL, double *ErrValF = NULL, unsigned long *ErrValU = NULL );
bool	TestForSNP( long MinProbMAQ, double MinProbBSNP, const PileupReader &Reader, const ShortReadGenotype &Caller, double &ProbSNP );
void	dump_debug( FILE *f, PileupReader &Reader, NonSNPBuffer &NSNPB );

int main( int argc, char *argv[] ){
	PileupReader		reader;					// Read a Samtools pileup file
	ShortReadGenotype	caller;					// This is the heart of BSNP, the genotype caller. Geenrates P(Geno | Reads ^ ReadQual ^ AlignQual ^ Params)
	InputArgs			input_args;				// User argument parser
	string				ShortReads, NucQual, AlignQual;
	char				HGRead;
	FILE				*fout_snp = NULL, *fout_nosnp = NULL, *fout_summary = NULL;
	bool				ok, done, hit_eof, is_snp_candidate;
	unsigned long		count;
	NonSNPBuffer		buf_nosnp;
	SummaryData			sum;
	double				num_trimmed[5];
	
	// Initialize debugging variables
	_debug_SNP_found = _debug_nSNP_found = _debug_SNP_written = _debug_LinesRead = _debug_LinesDiscarded= 0.0;
	for (int i=0; i<5;i++) num_trimmed[i] = 0.0;

	// Proces user arguments 
	ok = input_args.ParseArgs( argc, argv );
	if (!ok) {
		PrintUsage();
		return( -1 ); }

	// Initialize Priors
	{
		vector<double>	diploid_priors, haploid_priors;
		vector<string>  diploid_sate_names, haploid_state_names;
		caller.GetHaploidPriors(  haploid_state_names,	haploid_priors, input_args.phi );
		caller.GetDiploidPriors(  diploid_sate_names,	diploid_priors, input_args.phi, input_args.kappa, input_args.p0 );
		if (input_args.fileNameCustPrior.length() != 0) {
			ok = caller.ReadPriors( input_args.fileNameCustPrior, haploid_state_names,	haploid_priors,diploid_sate_names,	diploid_priors);
			if (!ok) return( -5 ); }
		caller.Init(	diploid_sate_names, diploid_priors, haploid_state_names, haploid_priors, input_args.sequencingTech );
	}

	// Open Input File Pileup
	ok = reader.FileOpen( input_args.fileNameIn );
	if (!ok) {
		LocalAbort( -3,"Cound Not Open Input File File",input_args.fileNameIn.c_str()); }

	// Open output file / SNP
	fout_snp = NULL;
	if ((input_args.fileNameOut == "-") || (input_args.fileNameOut == "") ) {
		fout_snp = stdout; 
	}else { 
		fout_snp = fopen( input_args.fileNameOut.c_str(), "w" ); };
	if (!fout_snp) {
		LocalAbort( -2,"Cound Not Open SNP Output File",input_args.fileNameOut.c_str()); }

	// SNP Output File / Print Header Line
	fprintf(fout_snp,"#ChrName\tChrPos\tRefCall\tMAQcall\t");
	if (input_args.printBSNPCall) {
		fprintf(fout_snp,"#BSNPcall\t"); }
	fprintf(fout_snp,"MAQcallQ\tMAQsnpQ\tMAQrmsQ\tP(Data)_NLL10");
	for (int i=0; i<10;i++) fprintf(fout_snp,"\tP(G=%s)",caller.DipName(i).c_str());
	for (int i=0; i<10;i++) fprintf(fout_snp,"\tP(D|G=%s)_NLL10",caller.DipName(i).c_str());
	fprintf(fout_snp,"\tNumBadReads\tNumGoodReads\tReads\tReadQual_Q\tAlignQual_Q\n");


	// Open output file / NoSNP
	fout_nosnp = NULL;
	if (input_args.dumpNonSNPs) {
		if ((input_args.fileNameOutNS == "-") || (input_args.fileNameOutNS == "") ) {
			fout_nosnp = stderr; // stderr?
		}else { 
			fout_nosnp = fopen( input_args.fileNameOutNS.c_str(), "w" ); };
		if (!fout_nosnp) {
			LocalAbort( -2,"Cound Not Open Non-SNP Output File",input_args.fileNameOutNS.c_str()); }
		// print header line
		fprintf(fout_nosnp,"#ChrName\tChrPos\tNumContigLoc\tNumValidReads\tProbSNP-5Lg10(p)\n");
	}

	// Open output file / Summary Info
	fout_summary = NULL;
	if (input_args.dumpSharedInfo) {
		if ((input_args.fileNameOutSum == "-") || (input_args.fileNameOutSum == "") ) {
			fout_summary = stdout; 
		} else { 
			fout_summary = fopen( input_args.fileNameOutSum.c_str(), "w" ); };
		if (!fout_summary) {
			LocalAbort( -2,"Cound Not Open Summary Output File",input_args.fileNameOutNS.c_str()); }
	}

	if (input_args.dumpSharedInfo) {
		fprintf(fout_summary,"# CMD: ");
		for (long i =0; i<argc; i++) fprintf(fout_summary," %s",argv[i]);
		fprintf(fout_summary,"\n");
		fflush( fout_summary );
		PrintSharedInfo( fout_summary, input_args, reader, caller ); 
	    fflush(fout_summary);  }

	// enter main loop
	done = false; count = 0;
	ok = reader.FetchNextLine( hit_eof );
	while (!done) {
		if (!ok) {
			done = true;
			if (hit_eof) break;	// this is a normal exit
			// fclose(fout); fout = NULL; reader.FileClose();
			LocalAbort( -4,"Error encountered while reading input file");}

		if (input_args.ignoreInvalidBases) {
			reader.TrimInvalid( &(num_trimmed[0])  ); }

		count++;
		if (input_args.verbose) {
			if (count%10000 == 0) { fprintf(stderr,"\r%ld",count); fflush(stderr); }	
			if (input_args.debugging) {
				if (count%100000 == 0) {
					if (input_args.debugging) { dump_debug( stderr, reader, buf_nosnp ); }; };
			};
		};

		reader.GetBases( ShortReads, NucQual, AlignQual, HGRead );

		ok = caller.SetData( ShortReads, NucQual, AlignQual, input_args.theta, input_args.thetaMin, input_args.invertTheta);
		// Some old code for external interference while debugging...
		if (!ok) {
			LocalAbort(-5,"Caller.SetData returns error. Likely corrupt Pileup File");  };

		// collect summary data on nucleotide dsitributions
		caller.GatherCountData( sum.nucsTotal, sum.nucsUnvoided, sum.nucsPartial, sum.nucsEffective );

		// collect summary data on genotype distributions.
		for (int i=0; i<10; i++) {
			sum.genoCounts[i]		+= caller.GetPosterior(i); };

		sum.dataLogLikelihood	+= caller.PDataNLL(); 

		// dump results, SNP and (if requested) NonSNP.
		if (ok) {
			double	prob_snp = 0.0;
			is_snp_candidate = TestForSNP( input_args.minMAQsnpQ, input_args.minBSNPprob, reader, caller, prob_snp );
			sum.callerLinesProcessed++;
			if ( is_snp_candidate) { sum.callerLinesSNP++;  _debug_SNP_found++; };
			if (!is_snp_candidate) { sum.callerLinesnSNP++; _debug_nSNP_found++; };
			if (caller.EffectiveReads() == 0.0)
				sum.callerLinesVoided++;
		    if (! input_args.dumpNonSNPs) {
				if (is_snp_candidate) {
					ok = PrintSNP( fout_snp, reader, caller, input_args );
					if (!ok) LocalAbort(-6,"Error Printing SNP likely fprintf failure."); }
			} else {
				if (is_snp_candidate) {
					ok = buf_nosnp.Flush( fout_nosnp );
					if (!ok) LocalAbort(-7,"Error 1 - flushing NoSnP file. Likely fprintf failure."); 
					ok = PrintSNP( fout_snp, reader, caller, input_args );
					if (!ok) LocalAbort(-6,"Error Printing SNP likely fprintf failure.");  
				} else {
					// In this case NonSNP means insufficient evidence to indicate a SNP. Not necessarily a 
					// Confirmation of a match to HG18.
					ok = buf_nosnp.Gather( fout_nosnp, reader.ReferenceSegmentName(), reader.ReferenceSegmentPos(), reader.NumReads(), prob_snp ); 
					if (!ok) LocalAbort(-8,"Error - Gathering noSNP. Likely fprintf failure."); 
				};
			};
		};

		ok = reader.FetchNextLine( hit_eof );
	}

	// make sure to flush the NSNP buffer if we've been collecting it...
	if (input_args.dumpNonSNPs) {
		buf_nosnp.Flush( fout_nosnp ); }

	sum.pileupLinesRead			= reader.D_LinesRead();
	sum.pileupLinesSkipped		= reader.D_LinesSkipped();
	sum.pileupLinesProcessed	= sum.pileupLinesRead - sum.pileupLinesSkipped;

	// add trimmed nucs back to total count!
	for (int i=0; i<5; i++)
		sum.nucsTotal[i] += num_trimmed[i];

	if (input_args.debugging) {
		dump_debug( stderr, reader, buf_nosnp ); 
		dump_debug( fout_summary, reader, buf_nosnp ); 
	}

	sum.Dump( fout_summary, caller );
	
	SAFE_CLOSE( fout_summary ); 

	//
	reader.FileClose();
	SAFE_CLOSE( fout_snp ); 	
	SAFE_CLOSE( fout_nosnp ); 	
	if (input_args.verbose) fprintf(stderr, "\nAll Done\n");
	// getchar();
	return( 0 );
}


void 
dump_debug( FILE *f, PileupReader &Reader, NonSNPBuffer &NSNPB ) {
	fprintf( f, "\n %10.0f:Read %10.0f:Discarded %10.0f:Used %10.0f:SNPF %10.0f:NSNPF %10.0f:SNPW %10.0f:NSNPT %10.0f:NSNPB\n",
			Reader.D_LinesRead(), Reader.D_LinesSkipped(),Reader.D_LinesRead()-Reader.D_LinesSkipped(),
			_debug_SNP_found, _debug_nSNP_found,
			_debug_SNP_written, NSNPB.D_NumTot(), NSNPB.D_NumBuff());
	// tests
	fprintf( f, " %10.0f:A %10.0f:B %10.0f:C %10.0f:D\n",
			Reader.D_LinesRead()-Reader.D_LinesSkipped()-_debug_SNP_found-_debug_nSNP_found,		// Num processed = found_SNP + found_nosnp
		    _debug_SNP_found   - _debug_SNP_written,                                                // Num SNP found = Num SNp Written
			_debug_nSNP_found  - NSNPB.D_NumBuff(),                                                 // Num nSNP found = Num nSNP Buffered
			NSNPB.D_NumBuff()  - NSNPB.D_NumTot() );                                                // Num ever buffered = Num Written + Num in current Buffer

return; 
}

bool
PrintSharedInfo(FILE *fout, const InputArgs &I, const PileupReader &Reader, ShortReadGenotype &Caller ){
	int i;

	// Print input args
	fprintf(fout,"#Input Args v%s :\n",VERSION);
	PrintUsage( &I, fout );

	fprintf(fout,"\n");
	// Print Priors
	fprintf(fout,"HaploidPriors:");
	for ( i=0;i<4;i++) fprintf(fout," %s:%e",Caller.HapName(i).c_str(),Caller.HapPrior(i));
	fprintf(fout,"\n");
	fprintf(fout,"DiploidPriors:");
	for ( i=0;i<10;i++) fprintf(fout," %s:%e",Caller.DipName(i).c_str(),Caller.DipPrior(i));
	fprintf(fout,"\n");
	// Print HG18 factors
	{
		long hg18_qual = (long) (-10.0*log10( I.p0 ));
		char hg18_qual_char[2]	= {0, 0};
		char hg18_base_char[2]	= {0, 0};
		char hg18_aln_char[2]	= {0, 0};
		char base_chars[5]			= { 'A', 'C', 'G', 'T', 0};
		string bases, base_qual, align_qual;

		if ( hg18_qual > (long) ('~' - '!')) hg18_qual = (long) ('~' - '!');
		hg18_qual_char[0]	= (char) ( hg18_qual + '!' );		// Q = 1/1000 wuality
		hg18_aln_char[0]	= (char) '~';						// nearly perfect alignment

		fprintf(fout,"\nHG18SingleNucleotideFactors:\n");
		base_qual	= hg18_qual_char;
		align_qual	= hg18_aln_char;
		for (i=0; i<4; i++) {
			hg18_base_char[0] = base_chars[i];
			bases = hg18_base_char;
			Caller.SetData( bases, base_qual, align_qual, 1.0, 1.0, false);		// Calculate distributions over the single base. Set Theta = 1.0 to disable correlated error feature.
			fprintf(fout,"HG18 %s :\t",hg18_base_char);
			fprintf( fout,"PData:%10.4e\t", Caller.PDataNLL() );
			for (int j=0;j<10;j++) {
				fprintf( fout,"%s:%10.4e\t",Caller.DipName(j).c_str(),Caller.GetPosterior( j ) ); }
			fprintf(fout,"\n");
		};
	}
	{ 
		// use the argument to prevent a warning, we may need the reader in a future version fo the code.
     		double d = Reader.D_LinesRead();
 		d += 1.0;
	}
	return( true );
}

bool	
TestForSNP( long MinProbMAQ, double MinProbBSNP, const PileupReader &Reader, const ShortReadGenotype &Caller, double &ProbSNP ) {
	char	hg18, maq_char;
	long	hg18_i, hg18_geno_i;
	long	maq_char_qual, maq_snp_qual, maq_qual_rms;
	bool	is_candidate;

	hg18			= Reader.GetReferenceBase();								// get the hg18 base for this location
	hg18_i			= ShortReadGenotype::NucCharToIndex( hg18 );				// get the index for the hg18 base
	hg18_geno_i		= ShortReadGenotype::GetHomozygousDiploidIndex( hg18_i );	// get the genotype corresponding th a homozygous hg18 base
	ProbSNP			= (1.0 - Caller.GetPosterior( hg18_geno_i ));				// anything other than the homozygous canonical sequence is a SNP candidate

	Reader.GetMAQCalls( maq_char, maq_char_qual, maq_snp_qual, maq_qual_rms );
	// we are a candiate if either MAQ or BSNP indicates that a SNP is possible,
	is_candidate = ( maq_char != hg18) && (maq_snp_qual >= MinProbMAQ);			// MAQ sees a SNP (the character comparison may be superfolous
	is_candidate = is_candidate || (ProbSNP >=MinProbBSNP);						// BSNP sees a possible SNP, not withotu an HG18 anchor, there may be a lot of these.
	if (Reader.NumReads()<1) is_candidate = false;								// OK this is a heuristic. Must have SOME data to call a SNP.

	return( is_candidate  );
}
	


bool	
PrintSNP(FILE *fout, const PileupReader &Reader, const ShortReadGenotype &Caller, const InputArgs &Args){
	string	seq_nam		= Reader.ReferenceSegmentName();
	ulong	seq_pos		= Reader.ReferenceSegmentPos();
	char	ref_call	= Reader.GetReferenceBase();
	char	maq_call;
	long	q_con, q_snp, q_rms;
	long	ok;
	static const char state_rep[] = "AMRWCSYGKT";

	Reader.GetMAQCalls( maq_call, q_con, q_snp, q_rms );
	ok = fprintf( fout,"%s\t%10lu\t%c\t%c\t",seq_nam.c_str(), seq_pos, ref_call, maq_call );
	if (ok < 1) return( false );

	if (Args.printBSNPCall) {
		long i_max = -1; double max = -1;
		for (int i=0;i<10;i++) {
			if (Caller.GetPosterior( i ) > max) {
				max = Caller.GetPosterior( i ); i_max = i; }; };
		ok = fprintf( fout,"%c\t",state_rep[i_max] );
		if (ok < 1) return( false ); }

	ok = fprintf( fout,"%ld\t%ld\t%ld\t", q_con, q_snp, q_rms );
	if (ok < 1) return( false );

	ok = fprintf( fout,"%10.4e\t", Caller.PDataNLL() );
	if (ok < 1) return( false );
	for (int i=0;i<10;i++) {
		ok = fprintf( fout,"%10.4e\t", Caller.GetPosterior( i ) ); 
		if (ok < 1) return( false ); }

	for (int i=0;i<10;i++) {
		ok = fprintf( fout,"%10.4e\t", Caller.GetCondLikelihoodNLL( i ) ); 
		if (ok < 1) return( false ); }

	ok = fprintf(fout,"%ld\t%ld\t",Reader.NumReadsTrimmed(),Reader.NumReads());
	if (ok < 1) return( false );

	{
		string	reads, read_qual, aln_qual;
		char	ref;
		Reader.GetBases( reads, read_qual, aln_qual, ref );
		ok = fprintf(fout,"%s\t",reads.c_str() );		if (ok < 1) return( false );
		ok = fprintf(fout,"%s\t",read_qual.c_str() );	if (ok < 1) return( false );
		ok = fprintf(fout,"%s",aln_qual.c_str() );		if (ok < 1) return( false );
		ok = fprintf(fout,"\n");						if (ok < 1) return( false );
	}

    _debug_SNP_written++;

	return( true );
}


void	PrintUsage( const InputArgs *Args, FILE *F ) {
	InputArgs DefaultArgs;
	const InputArgs *A;

	A = Args;
	if (A==NULL) A = &DefaultArgs;
	if (A == NULL)	Args	= &DefaultArgs;
	if (F == NULL)	F		= stderr;
	fprintf(F,"# BSNP %s \n",VERSION);
	if  (F==stderr) {
		fprintf(F,"# Usage:\n");
		fprintf(F,"\tBSNP [-i S] [-o S] [-on S] [-os S] [-ph F] [-k F] [-p0 F] [-mq I] [-mp F] [-th F] [-tm F] [-si B] [-ns B] [-ig B] [-v B] [-d B] [-ti B] [-pb B]\n");
		fprintf(F,"\t\t(Do not use both Verbose (-v 1) and -on -, or it will corrupt the NonSNP info in the stderr stream)\n");
		fprintf(F,"\t\tIt is highly reccomended that output be sent to files, rather than streams, so long as there is enough disk space.\n");
		fprintf(F,"\t\tExpected Input File is generated by Samtools Pileup -cv command.\n");
		fprintf(F,"\t\t\tSNP     file contains genotype probability distribution for candidate Polymorphisms.\n");
		fprintf(F,"\t\t\tnSNP    file contains ters summary info for each position confidently called to match reference sequence..\n");
		fprintf(F,"\t\t\tsummary file contains terse dataset summary information.\n");
	}
	fprintf(F,"\t\t-i  : Input File              : %s <- = stdin>\n",	A->fileNameIn.c_str() );
	fprintf(F,"\t\t-o  : Output File             : %s <- = stdout>\n",	A->fileNameOut.c_str() );
	fprintf(F,"\t\t-on : Output File (NonSNPInfo): %s <- = stderr>\n",	A->fileNameOutNS.c_str() );
	fprintf(F,"\t\t-os : Output File (SummryInfo): %s <- = stdout>\n",	A->fileNameOutSum.c_str() );
	fprintf(F,"\t\t-ph : Phee                    : %f\n", (double)		A->phi );
	fprintf(F,"\t\t-ka : Kappa                   : %f\n", (double)		A->kappa );
	fprintf(F,"\t\t-p0 : P0 (aka Pi, het Prob)   : %f\n", (double)		A->p0);
	fprintf(F,"\t\t-ip : Genotype prior override : %s <- <unused by default>\n", A->fileNameCustPrior.c_str() );
	fprintf(F,"\t\t-mq : Min MAQ SNP             : %d\n",  (int)		A->minMAQsnpQ);
	fprintf(F,"\t\t-mp : Min B SNP Probability   : %f\n", (double)		A->minBSNPprob);
	fprintf(F,"\t\t-th : Theta (0-1.0,1.0=off)   : %f\n", (double)		A->theta);
	fprintf(F,"\t\t-tm : ThetaMin (0-1.0,0=off)  : %f\n", (double)		A->thetaMin);
	fprintf(F,"\t\t-si : Dump Summary Info       : %d\n",  (int)		(A->dumpSharedInfo?1:0) );
	fprintf(F,"\t\t-ns : Dump NonSNP Info        : %d\n",  (int)		(A->ignoreInvalidBases?1:0) );
	fprintf(F,"\t\t-ig : Ignore Invalid Bases    : %d\n",  (int)		(A->ignoreInvalidBases?1:0) );
	fprintf(F,"\t\t-v  : Verbose Operation       : %d\n",  (int)		(A->verbose?1:0) );
	fprintf(F,"\t\t-d  : Dump Debugging Info     : %d\n",  (int)		(A->debugging?1:0) );
	fprintf(F,"\t\t-ti : Invert Theta Sorting    : %d\n",  (int)		(A->invertTheta?1:0) );
	fprintf(F,"\t\t-pb : Print BSNP Call Char    : %d\n",  (int)		(A->printBSNPCall?1:0) );
	fprintf(F,"\t\t-st : Sequencing Technology   : %d\t 0 = UNKNOWN, 1 = Illumina\\Solexa, 2 = Sanger, 3 = 454, 4 = SOLiD (converted to Nuc Space)\n",  (int)		(A->sequencingTech) );
	return;
}

void LocalAbort(long Code, const char *ErrMsg, const char *ErrContext , long *ErrValL , double *ErrValF , unsigned long *ErrValU ) {
	int	do_exit = 1;	// change this to resume when debugging;

	fprintf(stderr,"BSNP - terminal error\n");
	fprintf(stderr,"Code      : %ld\n",Code );
	fprintf(stderr,"Error Msg : %s\n" ,ErrMsg );
	if (ErrContext) {
		fprintf(stderr,"Context   : %s\n" ,ErrContext); }
	if (ErrValL) {
		fprintf(stderr,"Value     : %ld\n" ,*ErrValL); }
	if (ErrValF) {
		fprintf(stderr,"Value     : %f\n" ,(double) (*ErrValF)); }
	if (ErrValU) {
		fprintf(stderr,"Value     : %lu\n" ,*ErrValU); }

	if (do_exit) exit( (int) Code );

	return;
}
 
